<?php

namespace Installer\Traits;

use Exception;
use App\Models\Setting;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Installer\Models\Configuration;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Session;

trait InstallerMethods {
    private function checkMinimumRequirements(): array {
        $checks = [
            // Base requirements
            'php_version'         => [
                'check'   => PHP_VERSION_ID >= 80100,
                'message' => 'PHP version 8.1.0 is required. Current version: ' . PHP_VERSION,
                'url'     => 'https://www.php.net/releases/8.1/en.php',
            ],

            'extension_bcmath'    => [
                'check'   => extension_loaded( 'bcmath' ),
                'message' => 'The "bcmath" extension is required.',
                'url'     => 'https://www.php.net/manual/en/book.bc.php',
            ],

            'extension_ctype'     => [
                'check'   => extension_loaded( 'ctype' ),
                'message' => 'The "ctype" extension is required.',
                'url'     => 'https://www.php.net/manual/en/book.ctype.php',
            ],
            'extension_fileinfo'  => [
                'check'   => extension_loaded( 'fileinfo' ),
                'message' => 'The "fileinfo" extension is recommended for file features.',
                'url'     => 'https://www.php.net/manual/en/book.fileinfo.php',
            ],

            'extension_json'      => [
                'check'   => extension_loaded( 'json' ),
                'message' => 'The "json" extension is required.',
                'url'     => 'https://www.php.net/manual/en/book.json.php',
            ],

            'extension_mbstring'  => [
                'check'   => extension_loaded( 'mbstring' ),
                'message' => 'The "mbstring" extension is required.',
                'url'     => 'https://www.php.net/manual/en/book.mbstring.php',
            ],

            'extension_openssl'   => [
                'check'   => extension_loaded( 'openssl' ),
                'message' => 'The "openssl" extension is required.',
                'url'     => 'https://www.php.net/manual/en/book.openssl.php',
            ],

            'extension_pdo_mysql' => [
                'check'   => extension_loaded( 'pdo_mysql' ),
                'message' => 'The "pdo_mysql" extension is required for MySQL database access.',
                'url'     => 'https://www.php.net/manual/en/ref.pdo-mysql.php',
            ],

            'extension_tokenizer' => [
                'check'   => extension_loaded( 'tokenizer' ),
                'message' => 'The "tokenizer" extension is required.',
                'url'     => 'https://www.php.net/manual/en/book.tokenizer.php',
            ],

            'extension_xml'       => [
                'check'   => extension_loaded( 'xml' ),
                'message' => 'The "xml" extension is required.',
                'url'     => 'https://www.php.net/manual/en/book.simplexml.php',
            ],

            // File and directory permissions
            'env_writable'        => [
                'check'   => File::isWritable( base_path( '.env' ) ),
                'message' => 'The ".env" file must be writable.',
            ],

            'storage_writable'    => [
                'check'   => File::isWritable( storage_path() ) && File::isWritable( storage_path( 'logs' ) ),
                'message' => 'The "storage" and "storage/logs" directories must be writable.',
            ],
        ];

        $failedChecks = [];
        foreach ( $checks as $name => $check ) {
            if ( !$check['check'] ) {
                $failedChecks[$name] = [
                    'message' => $check['message'],
                    'url'     => isset( $check['url'] ) ? $check['url'] : null,
                ];
            }
        }

        $success = empty( $failedChecks );

        return [$checks, $success, $failedChecks];
    }
    private function requirementsCompleteStatus() {
        $success = $this->checkMinimumRequirements();
        return $success[1];
    }

    private function createDatabaseConnection( $details ) {
        try {
            Artisan::call( 'config:clear' );

            $defaultConnectionName = config( 'database.default' );
            Config::set( "database.connections.$defaultConnectionName.host", $details['host'] );
            Config::set( "database.connections.$defaultConnectionName.port", $details['port'] );
            Config::set( "database.connections.$defaultConnectionName.database", null );
            Config::set( "database.connections.$defaultConnectionName.username", $details['user'] );
            Config::set( "database.connections.$defaultConnectionName.password", $details['password'] );

            $customDatabase = $details['database'];

            DB::reconnect( $defaultConnectionName );

            $databaseExists = DB::connection()->select( 'SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = ?', [$customDatabase] );

            DB::purge( $defaultConnectionName );

            if ( empty( $databaseExists ) ) {
                Config::set( "database.connections.$defaultConnectionName.database", $customDatabase );
                DB::reconnect();

                return 'not-found';
            }
            Config::set( "database.connections.$defaultConnectionName.database", $customDatabase );
            DB::reconnect();

            if ( count( Schema::getAllTables() ) > 0 ) {
                if ( !empty( $details['reset_database'] ) && $details['reset_database'] == 'on' ) {
                    Schema::dropAllTables();
                    return true;
                }
                return 'table-exist';
            }

            return true;
        } catch ( Exception $e ) {
            Log::error( $e );

            return 'Database connection failed! Look like you have entered wrong database credentials (host, port, database, user or password).';
        }
    }

    private function importDatabase( $database_path ) {
        if ( File::exists( $database_path ) ) {
            try {
                DB::unprepared( File::get( $database_path ) );

                return true;
            } catch ( Exception $e ) {
                Log::error( $e );

                return 'Migration failed! Something went wrong';
            }
        } else {
            return 'Something went wrong';
        }
    }

    private function changeEnvDatabaseConfig( $config ) {
        changeEnvValues( 'DB_HOST', $config['host'] );
        changeEnvValues( 'DB_PORT', $config['port'] );
        changeEnvValues( 'DB_DATABASE', $config['database'] );
        changeEnvValues( 'DB_USERNAME', $config['user'] );
        changeEnvValues( 'DB_PASSWORD', $config['password'] );
    }

    private function completedSetup( $type ) {
        Configuration::updateCompeteStatus( 1 );
        Session::flush();
        Artisan::call( 'cache:clear' );
        if ( $type == 'admin' ) {
            return redirect()->route( 'admin.login' );
        } else {
            return redirect()->route( 'home' );
        }
    }
}
