<?php

namespace Installer\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\EmailConfiguration;
use Closure;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Installer\Enums\InstallerInfo;
use Installer\Models\Configuration;
use Installer\Traits\InstallerMethods;

class InstallerController extends Controller {
    use InstallerMethods;

    protected $requirementStatus;
    public function __construct() {
        set_time_limit( 8000000 );
        $this->requirementStatus = $this->checkMinimumRequirements();
        $this->middleware( function ( Request $request, Closure $next ) {
            $data = purchaseVerificationHashed( InstallerInfo::getLicenseFilePath() );
            if ( isset( $data ) && isset( $data['success'] ) && $data['success'] ) {
                return $next( $request );
            } else {
                if ( !strtolower( config( 'app.app_mode' ) ) ) {
                    return $next( $request );
                }
            }

            return redirect()->route( 'setup.verify' )->withInput()->withErrors( ['errors' => isset( $data ) && isset( $data['success'] ) && $data['message'] ? $data['message'] : 'License key not found'] );
        } );
    }

    public function requirements() {
        [$checks, $success, $failedChecks] = $this->checkMinimumRequirements();
        if ( $step = Configuration::stepExists( ) && $success) {
            if($step == 5){
                return redirect()->route( 'setup.complete' );
            }
            return redirect()->route( 'setup.account' );
        }
        session()->put( 'step-2-complete', true );
        return view( 'setup.requirements', compact( 'checks', 'success', 'failedChecks' ) );
    }

    public function database() {
        if ( $this->requirementsCompleteStatus() ) {
            session()->put( 'requirements-complete', true );
            if ( Configuration::stepExists() ) {
                return redirect()->route( 'setup.account' );
            }
            changeEnvValues( 'DB_DATABASE', '' );

            return view( 'setup.database', ['isLocalHost' => InstallerInfo::isRemoteLocal()] );
        }

        return redirect()->route( 'setup.requirements' )->withInput()->withErrors( ['errors' => 'Your server does not meet the minimum requirements.'] );
    }

    public function databaseSubmit( Request $request ) {
        if ( !$this->requirementsCompleteStatus() ) {
            return redirect()->route( 'setup.requirements' )->withInput()->withErrors( ['errors' => 'Your server does not meet the minimum requirements.'] );
        }

        try {
            $request->validate( [
                'host'     => 'required|ip',
                'port'     => 'required|integer',
                'database' => 'required',
                'user'     => 'required',
            ] );

            if ( !InstallerInfo::isRemoteLocal() ) {
                $request->validate( [
                    'password' => 'required',
                ] );
            }

            $databaseCreate = $this->createDatabaseConnection( $request->all() );

            if ( $databaseCreate !== true ) {
                if ( $databaseCreate == 'not-found' ) {
                    return response()->json( ['create_database' => true, 'message' => 'Database not found! Please create the database first.'], 200 );
                } elseif ( $databaseCreate == 'table-exist' ) {
                    return response()->json( ['reset_database' => true, 'message' => 'This database has tables already. Please create a new database or reset existing tables first to continue'], 200 );
                } else {
                    return response()->json( ['success' => false, 'message' => $databaseCreate], 200 );
                }
            }

            if ( $request->has( 'fresh_install' ) && $request->filled( 'fresh_install' ) && $request->fresh_install == 'on' ) {
                $migration = $this->importDatabase( InstallerInfo::getFreshDatabaseFilePath() );
                Cache::put( 'fresh_install', true, now()->addMinutes( 60 ) );
            } else {
                $migration = $this->importDatabase( InstallerInfo::getDummyDatabaseFilePath() );
            }

            if ( $migration !== true ) {
                return response()->json( ['success' => false, 'message' => $migration], 200 );
            }

            $this->changeEnvDatabaseConfig( $request->except( 'reset_database' ) );

            Cache::forget( 'fresh_install' );

            session()->put( 'step-3-complete', true );

            return response()->json( ['success' => true, 'message' => 'Successfully setup the database'], 200 );
        } catch ( Exception $e ) {
            Cache::forget( 'fresh_install' );
            return response()->json( ['success' => false, 'message' => 'Database connection failed! Look like you have entered wrong database credentials (host, port, database, user or password).'], 200 );
        }
    }

    public function account() {
        $step = Configuration::stepExists();
        if ( $step >= 1 && $step < 5 && $this->requirementStatus[1]) {
            $admin = $step >= 2 ? Admin::where('id',1)->select('name','email')->first() : null;
            return view( 'setup.account',compact('admin') );
        }
        if($step == 5){
            return redirect()->route( 'setup.requirements' );
        }
        return redirect()->route( 'setup.database' );
    }

    public function accountSubmit( Request $request ) {
        try {
            $request->validate( [
                'name'     => 'required',
                'email'    => 'required|email',
                'password' => 'required|same:confirm_password',
            ] );

            $admin = Admin::where( 'id', 1 )->where( 'admin_type', 1 )->first();
            if ( $admin ) {
                $admin->name = $request->name;
                $admin->email = $request->email;
                $admin->password = Hash::make( $request->password );
                $admin->save();
            } else {
                Admin::create( [
                    'name'       => $request->name,
                    'admin_type' => 1,
                    'email'      => $request->email,
                    'password'   => Hash::make( $request->password ),
                    'image'      => 'uploads/website-images/john-doe-2023-03-05-01-05-25-5417.jpg',
                ] );
            }

            Configuration::updateStep( 2 );
            session()->put( 'step-4-complete', true );

            return response()->json( ['success' => true, 'message' => 'Admin Account Successfully Created'], 200 );
        } catch ( Exception $e ) {
            return response()->json( ['success' => false, 'message' => 'Failed to Create Admin Account'], 200 );
        }
    }

    public function configuration() {
        $step = Configuration::stepExists();
        if($step == 5 || !$this->requirementStatus[1]){
            return redirect()->route( 'setup.requirements' )->withInput()->withErrors( ['errors' => 'Your server does not meet the minimum requirements.'] );
        }
        if ( $step < 2 ) {
            return redirect()->route( 'setup.account' );
        }
        $app_name = $step >= 3 ? config('app.name') : null;
        return view( 'setup.config',compact('app_name') );
    }

    public function configurationSubmit( Request $request ) {
        try {
            $request->validate( [
                'config_app_name' => 'required|string',
            ] );

            Configuration::updateStep( 3 );

            changeEnvValues( 'APP_NAME', '"'.$request->config_app_name.'"' );

            Cache::forget( 'setting' );
            session()->put( 'step-5-complete', true );

            return response()->json( ['success' => true, 'message' => 'Configuration Successfully Saved'], 200 );
        } catch ( Exception $e ) {
            Log::error( $e );

            return response()->json( ['success' => false, 'message' => 'Configuration Failed'], 200 );
        }
    }

    public function smtp() {
        $step = Configuration::stepExists();
        if($step == 5 || !$this->requirementStatus[1]){
            return redirect()->route( 'setup.requirements' )->withInput()->withErrors( ['errors' => 'Your server does not meet the minimum requirements.'] );
        }
        if ( $step < 3 ) {
            return redirect()->route( 'setup.configuration' );
        }
        $email = $step >= 4 ? EmailConfiguration::where('mail_host','!=', 'mail_host')->where('smtp_username','!=','smtp_username')->where('smtp_password','!=','smtp_password')->first() : null;
        return view( 'setup.smtp',compact('email') );
    }
    public function smtpSetup( Request $request ) {
        try {
            $rules = [
                'mail_host'       => 'required',
                'email'           => 'required',
                'smtp_username'   => 'required',
                'smtp_password'   => 'required',
                'mail_port'       => 'required',
                'mail_encryption' => 'required',
            ];
            $customMessages = [
                'mail_host.required'       => 'Mail host is required',
                'email.required'           => 'Email is required',
                'smtp_username.required'   => 'Smtp username is required',
                'smtp_password.unique'     => 'Smtp password is required',
                'mail_port.required'       => 'Mail port is required',
                'mail_encryption.required' => 'Mail encryption is required',
            ];
            $this->validate( $request, $rules, $customMessages );

            $email = EmailConfiguration::first();
            $email->mail_host = $request->mail_host;
            $email->mail_port = $request->mail_port;
            $email->email = $request->email;
            $email->smtp_username = $request->smtp_username;
            $email->smtp_password = $request->smtp_password;
            $email->mail_encryption = $request->mail_encryption;
            $email->save();

            Configuration::updateStep( 4 );

            session()->put( 'step-6-complete', true );

            return response()->json( ['success' => true, 'message' => 'Successfully setup mail SMTP'], 200 );
        } catch ( Exception $e ) {
            return response()->json( ['success' => false, 'message' => 'Failed to Setup SMTP'], 200 );
        }
    }
    public function smtpSkip( ) {
        Configuration::updateStep( 4 );
        session()->put( 'step-6-complete', true );
        return redirect()->route( 'setup.complete' );
    }

    public function setupComplete() {
        session()->put( 'step-7-complete', true );
        if ( Configuration::setupStepCheck( 4 )  && $this->requirementStatus[1] ) {
            changeEnvValues( 'APP_DEBUG', 'false' );
            changeEnvValues( 'APP_ENV', 'production' );

            return view( 'setup.complete' );
        }
        if ( Configuration::setupStepCheck( 5 ) && $this->requirementStatus[1] ) {
            return $this->completedSetup( 'home' );
        }

        return redirect()->back()->withInput()->withErrors( ['errors' => 'Setup Is Incomplete'] );
    }

    public function launchWebsite( $type ) {
        if ( Configuration::setupStepCheck( 4 ) ) {
            Configuration::updateStep( 5 );
            return $this->completedSetup( $type );
        } elseif ( Configuration::setupStepCheck( 5 ) ) {
            return $this->completedSetup( $type );
        }

        return redirect()->back()->withInput()->withErrors( ['errors' => 'Setup Is Incomplete'] );
    }
}
